/**
 * SPDX-FileCopyrightText: 2022-2025 Espressif Systems (Shanghai) CO LTD
 *
 *  SPDX-License-Identifier: Apache-2.0
 */
#pragma once

#include <stdint.h>
#include "soc/soc.h"
#include "soc/chip_rev.h"

#ifdef __cplusplus
extern "C" {
#endif


/** Version register */

/** ECDSA_DATE_REG register
 *  Version control
 *  register
 */
#define ECDSA_DATE_REG (DR_REG_ECDSA_BASE + 0xfc)
/* ECDSA_DATE : R/W; bitpos: [28:0]; default: 35684752 (0x21FFB30) for rev 0.0;
 * ECDSA_DATE : R/W; bitpos: [28:0]; default: 37761312 (0x2403120) for rev 1.2;
 * ECDSA version control
 * register
 */
#define ECDSA_DATE    0x0FFFFFFFU
#define ECDSA_DATE_M  (ECDSA_DATE_V << ECDSA_DATE_S)
#define ECDSA_DATE_V  0x0FFFFFFFU
#define ECDSA_DATE_S  0

/**
 * @brief Get the correct value of a field according to the register version
 * @note  ESP32-H2 v1.2 updated the register ECDSA_DATE_REG to a new version,
 *        At the same time the value of some registers was changed
 *        This macro can help to get the correct value of a field according to the register version
 ** @param old: the value of the field for the old version where DATE == 0x021FFB30 (rev 0.0)
 *  @param new: the value of the field for the new version where DATE == 0x2403120 (rev 1.2)
 */
#define ECDSA_REG_GET_OFFSET(old, new) (REG_GET_FIELD(ECDSA_DATE_REG, ECDSA_DATE) >= 0x02403120 ? (new) : (old))

/** Configuration registers */

/** ECDSA_CONF_REG register
 *  ECDSA configure
 *  register
 */
#define ECDSA_CONF_REG (DR_REG_ECDSA_BASE + 0x4)

/* ECDSA_WORK_MODE : R/W;
 * bitpos: [0]; default: 0; for DATE == 0x21FFB30 (rev 0.0)
 * bitpos: [1:0]; default: 0; for DATE == 0x2403120 (rev 1.2)
 * The work mode bits of ECDSA Accelerator.
 * 0: Signature Generate Mode.
 * 1: Signature Verify Mode.
 * 2: Export public key Mode. (only available for DATE == 0x2403120 (rev 1.2))
 * 3: Invalid mode. (only available for DATE == 0x2403120 (rev 1.2))
 */
#define ECDSA_WORK_MODE  ECDSA_REG_GET_OFFSET(BIT(0), 0x00000003U)
#define ECDSA_WORK_MODE_M  (ECDSA_WORK_MODE_V << ECDSA_WORK_MODE_S)
#define ECDSA_WORK_MODE_V  ECDSA_REG_GET_OFFSET(0x00000001U, 0x00000003U)
#define ECDSA_WORK_MODE_S  0

/* ECDSA_ECC_CURVE : R/W;
 * bitpos: [1]; default: 0; for DATE == 0x21FFB30 (rev 0.0)
 * bitpos: [2]; default: 0; for DATE == 0x2403120 (rev 1.2)
 * The ecc curve select bit of ECDSA Accelerator.
 * 0: P-192.  1: P-256.
 */
#define ECDSA_ECC_CURVE  ECDSA_REG_GET_OFFSET(BIT(1), BIT(2))
#define ECDSA_ECC_CURVE_M  (ECDSA_ECC_CURVE_V << ECDSA_ECC_CURVE_S)
#define ECDSA_ECC_CURVE_V  0x00000001U
#define ECDSA_ECC_CURVE_S  ECDSA_REG_GET_OFFSET(1, 2)
/* ECDSA_SOFTWARE_SET_K : R/W; bitpos: [2]; default: 0;
 * The source of k select bit. 0: k is automatically generated by TRNG. 1:
 * k is written by
 * software.
 */
#define ECDSA_SOFTWARE_SET_K  ECDSA_REG_GET_OFFSET(BIT(2), BIT(3))
#define ECDSA_SOFTWARE_SET_K_M  (ECDSA_SOFTWARE_SET_K_V << ECDSA_SOFTWARE_SET_K_S)
#define ECDSA_SOFTWARE_SET_K_V  0x00000001U
#define ECDSA_SOFTWARE_SET_K_S  ECDSA_REG_GET_OFFSET(2, 3)

/* ECDSA_SOFTWARE_SET_Z : R/W; bitpos: [3]; default: 0;
 * The source of z select bit. 0: z is generated from SHA result. 1: z is
 * written by
 * software.
 */
#define ECDSA_SOFTWARE_SET_Z  ECDSA_REG_GET_OFFSET(BIT(3), BIT(4))
#define ECDSA_SOFTWARE_SET_Z_M  (ECDSA_SOFTWARE_SET_Z_V << ECDSA_SOFTWARE_SET_Z_S)
#define ECDSA_SOFTWARE_SET_Z_V  0x00000001U
#define ECDSA_SOFTWARE_SET_Z_S  ECDSA_REG_GET_OFFSET(3, 4)

/* ECDSA_K_DETERMINISTIC : R/W; bitpos: [5]; default: 0;
 * The source of k select bit. 0: k is generated from TRNG. 1: k is
 * written by
 * software.
 */
#define ECDSA_DETERMINISTIC_K    (BIT(5))
#define ECDSA_DETERMINISTIC_K_M  (ECDSA_DETERMINISTIC_K_V << ECDSA_DETERMINISTIC_K_S)
#define ECDSA_DETERMINISTIC_K_V  0x00000001U
#define ECDSA_DETERMINISTIC_K_S  5


/** Clock and reset registers */

/** ECDSA_CLK_REG register
 *  ECDSA clock gate
 *  register
 */
#define ECDSA_CLK_REG (DR_REG_ECDSA_BASE + 0x8)
/* ECDSA_CLK_GATE_FORCE_ON : R/W; bitpos: [0]; default: 0;
 * Write 1 to force on register clock
 * gate.
 */
#define ECDSA_CLK_GATE_FORCE_ON    (BIT(0))
#define ECDSA_CLK_GATE_FORCE_ON_M  (ECDSA_CLK_GATE_FORCE_ON_V << ECDSA_CLK_GATE_FORCE_ON_S)
#define ECDSA_CLK_GATE_FORCE_ON_V  0x00000001U
#define ECDSA_CLK_GATE_FORCE_ON_S  0


/** Interrupt registers */

/** ECDSA_INT_RAW_REG register
 *  ECDSA interrupt raw register, valid in
 *  level.
 */
#define ECDSA_INT_RAW_REG (DR_REG_ECDSA_BASE + 0xc)

/* ECDSA_PREP_DONE_INT_RAW : RO/WTC/SS; bitpos: [0]; default: 0;
 * The raw interrupt status bit  for the ecdsa_prep_done_int
 * interrupt
 * This bit was named as ECDSA_CALC_DONE_INT_RAW in rev 0.0 and changed to ECDSA_PREP_DONE_INT_RAW in rev 1.2
 * functionality is the same
 */
#define ECDSA_PREP_DONE_INT_RAW    (BIT(0))
#define ECDSA_PREP_DONE_INT_RAW_M  (ECDSA_PREP_DONE_INT_RAW_V << ECDSA_PREP_DONE_INT_RAW_S)
#define ECDSA_PREP_DONE_INT_RAW_V  0x00000001U
#define ECDSA_PREP_DONE_INT_RAW_S  0

/* ECDSA_PROC_DONE_INT_RAW : RO/WTC/SS; bitpos: [1]; default: 0;
 * The raw interrupt status bit  for the ecdsa_proc_done_int
 * interrupt
 * This bit is only available for DATE == 0x2403120 (rev 1.2)
 */
#define ECDSA_PROC_DONE_INT_RAW    (BIT(1))
#define ECDSA_PROC_DONE_INT_RAW_M  (ECDSA_PROC_DONE_INT_RAW_V << ECDSA_PROC_DONE_INT_RAW_S)
#define ECDSA_PROC_DONE_INT_RAW_V  0x00000001U
#define ECDSA_PROC_DONE_INT_RAW_S  1

/* ECDSA_POST_DONE_INT_RAW : RO/WTC/SS; bitpos: [2]; default: 0;
 * The raw interrupt status bit  for the ecdsa_post_done_int
 * interrupt
 * This bit is only available for DATE == 0x2403120 (rev 1.2)
 */
#define ECDSA_POST_DONE_INT_RAW    (BIT(2))
#define ECDSA_POST_DONE_INT_RAW_M  (ECDSA_POST_DONE_INT_RAW_V << ECDSA_POST_DONE_INT_RAW_S)
#define ECDSA_POST_DONE_INT_RAW_V  0x00000001U
#define ECDSA_POST_DONE_INT_RAW_S  2

/* ECDSA_SHA_RELEASE_INT_RAW : RO/WTC/SS; bitpos: [1]; default: 0;
 * The raw interrupt status bit  for the ecdsa_sha_release_int
 * interrupt
 * This bit is only available for DATE == 0x2403120 (rev 1.2)
 */
#define ECDSA_SHA_RELEASE_INT_RAW  ECDSA_REG_GET_OFFSET(BIT(1), BIT(3))
#define ECDSA_SHA_RELEASE_INT_RAW_M  (ECDSA_SHA_RELEASE_INT_RAW_V << ECDSA_SHA_RELEASE_INT_RAW_S)
#define ECDSA_SHA_RELEASE_INT_RAW_V  0x00000001U
#define ECDSA_SHA_RELEASE_INT_RAW_S  ECDSA_REG_GET_OFFSET(1, 3)

/** ECDSA_INT_ST_REG register
 *  ECDSA interrupt status
 *  register.
 */
#define ECDSA_INT_ST_REG (DR_REG_ECDSA_BASE + 0x10)

/* ECDSA_PREP_DONE_INT_ST : RO; bitpos: [0]; default: 0;
 * The masked interrupt status bit  for the ecdsa_prep_done_int
 * interrupt
 * This bit was named as ECDSA_CALC_DONE_INT_ST in rev 0.0 and changed to ECDSA_PREP_DONE_INT_ST in rev 1.2
 * functionality is the same
 */
#define ECDSA_PREP_DONE_INT_ST    (BIT(0))
#define ECDSA_PREP_DONE_INT_ST_M  (ECDSA_PREP_DONE_INT_ST_V << ECDSA_PREP_DONE_INT_ST_S)
#define ECDSA_PREP_DONE_INT_ST_V  0x00000001U
#define ECDSA_PREP_DONE_INT_ST_S  0

/* ECDSA_PROC_DONE_INT_ST : RO; bitpos: [1]; default: 0;
 * The masked interrupt status bit  for the ecdsa_proc_done_int
 * interrupt
 * This bit is only available for DATE == 0x2403120 (rev 1.2)
 */
#define ECDSA_PROC_DONE_INT_ST    (BIT(1))
#define ECDSA_PROC_DONE_INT_ST_M  (ECDSA_PROC_DONE_INT_ST_V << ECDSA_PROC_DONE_INT_ST_S)
#define ECDSA_PROC_DONE_INT_ST_V  0x00000001U
#define ECDSA_PROC_DONE_INT_ST_S  1

/* ECDSA_POST_DONE_INT_ST : RO; bitpos: [2]; default: 0;
 * The masked interrupt status bit  for the ecdsa_post_done_int
 * interrupt
 * This bit is only available for DATE == 0x2403120 (rev 1.2)
 */
#define ECDSA_POST_DONE_INT_ST    (BIT(2))
#define ECDSA_POST_DONE_INT_ST_M  (ECDSA_POST_DONE_INT_ST_V << ECDSA_POST_DONE_INT_ST_S)
#define ECDSA_POST_DONE_INT_ST_V  0x00000001U
#define ECDSA_POST_DONE_INT_ST_S  2

/* ECDSA_SHA_RELEASE_INT_ST : RO;
 * bitpos: [1] for DATE == 0x21FFB30 (rev 0.0)  ; default: 0;
 * bitpos: [3] for DATE == 0x2403120 (rev 1.2)  ; default: 0;
 * The masked interrupt status bit  for the ecdsa_sha_release_int
 * interrupt
 */
#define ECDSA_SHA_RELEASE_INT_ST  ECDSA_REG_GET_OFFSET(BIT(1), BIT(3))
#define ECDSA_SHA_RELEASE_INT_ST_M  (ECDSA_SHA_RELEASE_INT_ST_V << ECDSA_SHA_RELEASE_INT_ST_S)
#define ECDSA_SHA_RELEASE_INT_ST_V  0x00000001U
#define ECDSA_SHA_RELEASE_INT_ST_S  ECDSA_REG_GET_OFFSET(1, 3)

/** ECDSA_INT_ENA_REG register
 *  ECDSA interrupt enable
 *  register.
 */
#define ECDSA_INT_ENA_REG (DR_REG_ECDSA_BASE + 0x14)

/* ECDSA_PREP_DONE_INT_ENA : R/W; bitpos: [0]; default: 0;
 * The interrupt enable bit  for the ecdsa_prep_done_int
 * interrupt
 * This bit was named as ECDSA_CALC_DONE_INT_ENA in rev 0.0 and changed to ECDSA_PREP_DONE_INT_ENA in rev 1.2
 * functionality is the same
 */
#define ECDSA_PREP_DONE_INT_ENA    (BIT(0))
#define ECDSA_PREP_DONE_INT_ENA_M  (ECDSA_PREP_DONE_INT_ENA_V << ECDSA_PREP_DONE_INT_ENA_S)
#define ECDSA_PREP_DONE_INT_ENA_V  0x00000001U
#define ECDSA_PREP_DONE_INT_ENA_S  0

/* ECDSA_PROC_DONE_INT_ENA : R/W; bitpos: [1]; default: 0;
 * The interrupt enable bit  for the ecdsa_proc_done_int
 * interrupt
 * This bit is only available for DATE == 0x2403120 (rev 1.2)
 */
#define ECDSA_PROC_DONE_INT_ENA    (BIT(1))
#define ECDSA_PROC_DONE_INT_ENA_M  (ECDSA_PROC_DONE_INT_ENA_V << ECDSA_PROC_DONE_INT_ENA_S)
#define ECDSA_PROC_DONE_INT_ENA_V  0x00000001U
#define ECDSA_PROC_DONE_INT_ENA_S  1

/* ECDSA_POST_DONE_INT_ENA : R/W; bitpos: [2]; default: 0;
 * The interrupt enable bit  for the ecdsa_post_done_int
 * interrupt
 * This bit is only available for DATE == 0x2403120 (rev 1.2)
 */
#define ECDSA_POST_DONE_INT_ENA    (BIT(2))
#define ECDSA_POST_DONE_INT_ENA_M  (ECDSA_POST_DONE_INT_ENA_V << ECDSA_POST_DONE_INT_ENA_S)
#define ECDSA_POST_DONE_INT_ENA_V  0x00000001U
#define ECDSA_POST_DONE_INT_ENA_S  2

/* ECDSA_SHA_RELEASE_INT_ENA : R/W;
 * bitpos: [1] for DATE == 0x21FFB30 (rev 0.0); default: 0;
 * bitpos: [3] for DATE == 0x2403120 (rev 1.2); default: 0;
 * The interrupt enable bit  for the ecdsa_sha_release_int
 * interrupt
 */
#define ECDSA_SHA_RELEASE_INT_ENA    (ECDSA_REG_GET_OFFSET(BIT(1), BIT(3)))
#define ECDSA_SHA_RELEASE_INT_ENA_M  (ECDSA_SHA_RELEASE_INT_ENA_V << ECDSA_SHA_RELEASE_INT_ENA_S)
#define ECDSA_SHA_RELEASE_INT_ENA_V  0x00000001U
#define ECDSA_SHA_RELEASE_INT_ENA_S  (ECDSA_REG_GET_OFFSET(1, 3))

/** ECDSA_INT_CLR_REG register
 *  ECDSA interrupt clear
 *  register.
 */
#define ECDSA_INT_CLR_REG (DR_REG_ECDSA_BASE + 0x18)


/* ECDSA_PREP_DONE_INT_CLR : WT; bitpos: [0]; default: 0;
 * Set this bit to clear the ecdsa_prep_done_int interrupt
 * This bit was named as ECDSA_CALC_DONE_INT_CLR in rev 0.0 and changed to ECDSA_PREP_DONE_INT_CLR in rev 1.2
 * functionality is the same
 */
#define ECDSA_PREP_DONE_INT_CLR    (BIT(0))
#define ECDSA_PREP_DONE_INT_CLR_M  (ECDSA_PREP_DONE_INT_CLR_V << ECDSA_PREP_DONE_INT_CLR_S)
#define ECDSA_PREP_DONE_INT_CLR_V  0x00000001U
#define ECDSA_PREP_DONE_INT_CLR_S  0

#define ECDSA_PROC_DONE_INT_CLR    (BIT(1))
#define ECDSA_PROC_DONE_INT_CLR_M  (ECDSA_PROC_DONE_INT_CLR_V << ECDSA_PROC_DONE_INT_CLR_S)
#define ECDSA_PROC_DONE_INT_CLR_V  0x00000001U
#define ECDSA_PROC_DONE_INT_CLR_S  1

/* This bit is only available for DATE == 0x2403120 (rev 1.2)
 * ECDSA_POST_DONE_INT_CLR : WT; bitpos: [2]; default: 0;
 *  Set this bit to clear the ecdsa_post_done_int interrupt
 */

#define ECDSA_POST_DONE_INT_CLR    (BIT(2))
#define ECDSA_POST_DONE_INT_CLR_M  (ECDSA_POST_DONE_INT_CLR_V << ECDSA_POST_DONE_INT_CLR_S)
#define ECDSA_POST_DONE_INT_CLR_V  0x00000001U
#define ECDSA_POST_DONE_INT_CLR_S  2

/* ECDSA_SHA_RELEASE_INT_CLR : WT;
 * bitpos: [1] for DATE == 0x21FFB30 (rev 0.0); default: 0;
 * bitpos: [3] for DATE == 0x2403120 (rev 1.2); default: 0;
 * Set this bit to clear the ecdsa_sha_release_int
 * interrupt
 */
#define ECDSA_SHA_RELEASE_INT_CLR    (ECDSA_REG_GET_OFFSET(BIT(1), BIT(3)))
#define ECDSA_SHA_RELEASE_INT_CLR_M  (ECDSA_SHA_RELEASE_INT_CLR_V << ECDSA_SHA_RELEASE_INT_CLR_S)
#define ECDSA_SHA_RELEASE_INT_CLR_V  0x00000001U
#define ECDSA_SHA_RELEASE_INT_CLR_S  (ECDSA_REG_GET_OFFSET(1, 3))

/** ECDSA_START_REG register
 *  ECDSA start
 *  register
 */
#define ECDSA_START_REG (DR_REG_ECDSA_BASE + 0x1c)
/* ECDSA_START : WT; bitpos: [0]; default: 0;
 * Write 1 to start calculation of ECDSA Accelerator. This bit will be
 * self-cleared after
 * configuration.
 */
#define ECDSA_START    (BIT(0))
#define ECDSA_START_M  (ECDSA_START_V << ECDSA_START_S)
#define ECDSA_START_V  0x00000001U
#define ECDSA_START_S  0
/* ECDSA_LOAD_DONE : WT; bitpos: [1]; default: 0;
 * Write 1 to input load done signal of ECDSA Accelerator. This bit will
 * be self-cleared after
 * configuration.
 */
#define ECDSA_LOAD_DONE    (BIT(1))
#define ECDSA_LOAD_DONE_M  (ECDSA_LOAD_DONE_V << ECDSA_LOAD_DONE_S)
#define ECDSA_LOAD_DONE_V  0x00000001U
#define ECDSA_LOAD_DONE_S  1
/* ECDSA_GET_DONE : WT; bitpos: [2]; default: 0;
 * Write 1 to input get done signal of ECDSA Accelerator. This bit will be
 * self-cleared after
 * configuration.
 */
#define ECDSA_GET_DONE    (BIT(2))
#define ECDSA_GET_DONE_M  (ECDSA_GET_DONE_V << ECDSA_GET_DONE_S)
#define ECDSA_GET_DONE_V  0x00000001U
#define ECDSA_GET_DONE_S  2

/** Status registers */

/** ECDSA_STATE_REG register
 *  ECDSA status
 *  register
 */
#define ECDSA_STATE_REG (DR_REG_ECDSA_BASE + 0x20)
/* ECDSA_BUSY : RO; bitpos: [2:0]; default: 0;
 * The status bits of ECDSA Accelerator. ECDSA is at 0: IDLE, 1: LOAD, 2:
 * GET, 3: BUSY
 * state.
 */
#define ECDSA_BUSY    0x00000003U
#define ECDSA_BUSY_M  (ECDSA_BUSY_V << ECDSA_BUSY_S)
#define ECDSA_BUSY_V  0x00000003U
#define ECDSA_BUSY_S  0


/** Result registers */

/** ECDSA_RESULT_REG register
 *  ECDSA result
 *  register
 */
#define ECDSA_RESULT_REG (DR_REG_ECDSA_BASE + 0x24)
/* ECDSA_OPERATION_RESULT : RO/SS; bitpos: [0]; default: 0;
 * The operation result bit of ECDSA Accelerator, only valid when ECDSA
 * calculation is
 * done.
 */
#define ECDSA_OPERATION_RESULT    (BIT(0))
#define ECDSA_OPERATION_RESULT_M  (ECDSA_OPERATION_RESULT_V << ECDSA_OPERATION_RESULT_S)
#define ECDSA_OPERATION_RESULT_V  0x00000001U
#define ECDSA_OPERATION_RESULT_S  0


/** SHA register */

/** ECDSA_SHA_MODE_REG register
 *  ECDSA control SHA
 *  register
 */
#define ECDSA_SHA_MODE_REG (DR_REG_ECDSA_BASE + 0x200)
/* ECDSA_SHA_MODE : R/W; bitpos: [3:0]; default: 0;
 * The work mode bits of SHA Calculator in ECDSA Accelerator. 1: SHA-224.
 * 2: SHA-256. Others:
 * invalid.
 */
#define ECDSA_SHA_MODE    0x00000007U
#define ECDSA_SHA_MODE_M  (ECDSA_SHA_MODE_V << ECDSA_SHA_MODE_S)
#define ECDSA_SHA_MODE_V  0x00000007U
#define ECDSA_SHA_MODE_S  0

/** ECDSA_SHA_START_REG register
 *  ECDSA control SHA
 *  register
 */
#define ECDSA_SHA_START_REG (DR_REG_ECDSA_BASE + 0x210)
/* ECDSA_SHA_START : WT; bitpos: [0]; default: 0;
 * Write 1 to start the first calculation of SHA Calculator in ECDSA
 * Accelerator. This bit will be self-cleared after
 * configuration.
 */
#define ECDSA_SHA_START    (BIT(0))
#define ECDSA_SHA_START_M  (ECDSA_SHA_START_V << ECDSA_SHA_START_S)
#define ECDSA_SHA_START_V  0x00000001U
#define ECDSA_SHA_START_S  0

/** ECDSA_SHA_CONTINUE_REG register
 *  ECDSA control SHA
 *  register
 */
#define ECDSA_SHA_CONTINUE_REG (DR_REG_ECDSA_BASE + 0x214)
/* ECDSA_SHA_CONTINUE : WT; bitpos: [0]; default: 0;
 * Write 1 to start the latter calculation of SHA Calculator in ECDSA
 * Accelerator. This bit will be self-cleared after
 * configuration.
 */
#define ECDSA_SHA_CONTINUE    (BIT(0))
#define ECDSA_SHA_CONTINUE_M  (ECDSA_SHA_CONTINUE_V << ECDSA_SHA_CONTINUE_S)
#define ECDSA_SHA_CONTINUE_V  0x00000001U
#define ECDSA_SHA_CONTINUE_S  0

/** ECDSA_SHA_BUSY_REG register
 *  ECDSA status
 *  register
 */
#define ECDSA_SHA_BUSY_REG (DR_REG_ECDSA_BASE + 0x218)
/* ECDSA_SHA_BUSY : RO; bitpos: [0]; default: 0;
 * The busy status bit of SHA Calculator in ECDSA Accelerator. 1:SHA is in
 * calculation. 0: SHA is
 * idle.
 */
#define ECDSA_SHA_BUSY    (BIT(0))
#define ECDSA_SHA_BUSY_M  (ECDSA_SHA_BUSY_V << ECDSA_SHA_BUSY_S)
#define ECDSA_SHA_BUSY_V  0x00000001U
#define ECDSA_SHA_BUSY_S  0


/** ECDSA_MESSAGE_MEM register
 *  The memory that stores message.
 */
#define ECDSA_MESSAGE_MEM (DR_REG_ECDSA_BASE + 0x280)
#define ECDSA_MESSAGE_MEM_SIZE_BYTES 32

/** ECDSA_R_MEM register
 *  The memory that stores r.
 */
#define ECDSA_R_MEM (DR_REG_ECDSA_BASE + REG_COMPATIBLE_ADDR(102, 0xa00, 0x340))
#define ECDSA_R_MEM_SIZE_BYTES 32

/** ECDSA_S_MEM register
 *  The memory that stores s.
 */
#define ECDSA_S_MEM (DR_REG_ECDSA_BASE + REG_COMPATIBLE_ADDR(102, 0xa20, 0x360))
#define ECDSA_S_MEM_SIZE_BYTES 32

/** ECDSA_Z_MEM register
 *  The memory that stores software written z.
 */
#define ECDSA_Z_MEM (DR_REG_ECDSA_BASE + REG_COMPATIBLE_ADDR(102, 0xa40, 0x380))
#define ECDSA_Z_MEM_SIZE_BYTES 32

/** ECDSA_QAX_MEM register
 *  The memory that stores x coordinates of QA or software written k.
 */
#define ECDSA_QAX_MEM (DR_REG_ECDSA_BASE + REG_COMPATIBLE_ADDR(102, 0xa60, 0x3a0))
#define ECDSA_QAX_MEM_SIZE_BYTES 32

/** ECDSA_QAY_MEM register
 *  The memory that stores y coordinates of QA.
 */
#define ECDSA_QAY_MEM (DR_REG_ECDSA_BASE + REG_COMPATIBLE_ADDR(102, 0xa80, 0x3c0))
#define ECDSA_QAY_MEM_SIZE_BYTES 32

#ifdef __cplusplus
}
#endif
