/*
 * SPDX-FileCopyrightText: 2025 Espressif Systems (Shanghai) CO LTD
 *
 * SPDX-License-Identifier: Unlicense OR CC0-1.0
 */
#include <stdint.h>

#define TEST_COUNT 5

typedef struct test_xts_data {
    uint16_t data_size;
    uint32_t data_offset;
    uint8_t ciphertext[128];
} test_xts_data_t;

typedef struct test_ecdsa_data {
    uint8_t pubx[32];
    uint8_t puby[32];
} test_ecdsa_data_t;

typedef struct test_hmac_data {
    uint8_t message[257];
    uint8_t hmac_result[32];
} test_hmac_data_t;

typedef struct test_ds_data {
    uint8_t ds_message[3072 / 8];
    uint8_t ds_encrypted_input_params[1200];
    uint8_t ds_encrypted_input_params_iv[16];
    size_t ds_key_size;
    uint8_t ds_result[384];
} test_ds_data_t;

typedef struct test_data {
    uint8_t init_key[32];
    uint8_t k2_info[64];
    uint8_t k1_encrypted[2][32];  // For both 256-bit and 512-bit keys
    uint8_t plaintext_data[128];
    union {
        test_xts_data_t xts_test_data[TEST_COUNT];
        test_ecdsa_data_t ecdsa_test_data;
        test_hmac_data_t hmac_test_data;
        test_ds_data_t ds_test_data;
    };
} test_data_aes_mode_t;

typedef struct test_data_ecdh0 {
    uint8_t plaintext_data[128];
    uint8_t k1[2][32];
    uint8_t k1_G[2][64];
} test_data_ecdh0_mode_t;

// For 32-byte k1 key
test_data_aes_mode_t test_data_xts_aes_128 = {
    .init_key = { 0xee, 0x89, 0x95, 0xda, 0x3c, 0x8a, 0x43, 0x83, 0xa9, 0x4b, 0x25, 0x5b, 0x04, 0x7e, 0xf1, 0x57, 0xb8, 0xe8, 0x06, 0x45, 0x87, 0x76, 0xee, 0x1b, 0x4e, 0x2e, 0x55, 0xa7, 0x1f, 0x25, 0xe1, 0x94 },
    .k2_info = { 0x8f, 0x96, 0x33, 0x47, 0xe1, 0xa5, 0x57, 0xe9, 0x2a, 0x51, 0xa9, 0xbe, 0x48, 0x84, 0x25, 0x4e, 0x6f, 0x50, 0x1c, 0x45, 0xdb, 0xb6, 0xfa, 0xeb, 0x35, 0xd2, 0x27, 0x91, 0x3f, 0x67, 0x57, 0xd9, 0xcb, 0x55, 0xe4, 0x2b, 0x18, 0x16, 0xe7, 0xce, 0x6c, 0xf2, 0x58, 0x71, 0x17, 0x76, 0x2a, 0x86, 0x05, 0xe7, 0x37, 0x45, 0x71, 0x34, 0xca, 0xaf, 0x60, 0x07, 0xdf, 0xf4, 0xd2, 0xee, 0x3d, 0x4b },
    .k1_encrypted = { { 0xe0, 0xe8, 0x41, 0xe3, 0xd0, 0x92, 0x71, 0x84, 0x4b, 0x02, 0x1e, 0xec, 0x14, 0xdd, 0xaf, 0xf8, 0x39, 0xf9, 0x6a, 0x8d, 0x1b, 0xd7, 0x64, 0x3b, 0x7b, 0xa6, 0x05, 0x42, 0x01, 0xfb, 0xab, 0xe1 }, {  } },
    .plaintext_data = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80 },
    .xts_test_data = {
		{.data_size = 32, .data_offset = 0x160000, .ciphertext = {0x0d, 0x02, 0x33, 0x69, 0x2f, 0x0f, 0x6f, 0x3e, 0xd1, 0xf0, 0x3d, 0x38, 0x63, 0xe3, 0x45, 0xe1, 0x01, 0xe2, 0xde, 0x88, 0xf2, 0x4e, 0x94, 0xa2, 0x22, 0xfe, 0x01, 0x6e, 0xe0, 0xf5, 0x16, 0x7c}},
		{.data_size = 64, .data_offset = 0x160100, .ciphertext = {0xc0, 0xc8, 0x19, 0x93, 0x12, 0xa2, 0xa6, 0x9c, 0xeb, 0x2b, 0x15, 0x84, 0x06, 0x71, 0x34, 0xfc, 0xef, 0xba, 0x53, 0xef, 0x66, 0xd8, 0xfd, 0x7f, 0x47, 0x88, 0x03, 0xe7, 0x44, 0xc4, 0x83, 0x30, 0x11, 0x2d, 0xd8, 0x87, 0xcd, 0xf9, 0x0c, 0x74, 0xa4, 0x14, 0x2d, 0xa5, 0xab, 0xf6, 0xd7, 0xdc, 0x4f, 0x8d, 0x22, 0x1a, 0x2e, 0x3d, 0x6d, 0x0f, 0xb3, 0xed, 0xf0, 0x7b, 0x01, 0x18, 0xf0, 0xd3}},
		{.data_size = 128, .data_offset = 0x160200, .ciphertext = {0xba, 0xe8, 0x7d, 0xfe, 0x1d, 0x7c, 0x95, 0x41, 0x5b, 0x59, 0x84, 0x4b, 0x37, 0x8e, 0x29, 0x53, 0xf5, 0x9d, 0x90, 0x07, 0xec, 0xc9, 0xdf, 0x52, 0xd5, 0xab, 0x7c, 0x73, 0x21, 0x52, 0x8d, 0xdc, 0x6f, 0xe1, 0xaa, 0x16, 0x4d, 0x86, 0x8a, 0x12, 0x29, 0x49, 0x9f, 0x96, 0x23, 0xd2, 0x4c, 0xa8, 0xcf, 0xe7, 0xa8, 0x83, 0x69, 0x57, 0x41, 0x92, 0x0a, 0x06, 0xf8, 0x7a, 0x30, 0xc6, 0xd6, 0x51, 0xb0, 0x34, 0x46, 0x08, 0x77, 0xc9, 0x49, 0x9d, 0x63, 0xee, 0x9f, 0x66, 0x08, 0xc1, 0x01, 0x0c, 0x07, 0x24, 0xc2, 0x76, 0x86, 0x14, 0xcb, 0xa1, 0x27, 0xc0, 0xe9, 0xcd, 0x1d, 0x60, 0x70, 0xa0, 0x0a, 0x21, 0x9e, 0x91, 0xfa, 0x1a, 0x8c, 0x10, 0x87, 0x17, 0x36, 0xf6, 0x20, 0xc2, 0x7e, 0x96, 0x0f, 0xde, 0x30, 0x28, 0x5a, 0x3a, 0x9e, 0x08, 0xe1, 0x35, 0xb3, 0x36, 0x2f, 0xc7, 0x0d, 0x28}},
		{.data_size = 16, .data_offset = 0x160300, .ciphertext = {0x0a, 0x2c, 0xcf, 0x75, 0x73, 0xa0, 0x5f, 0x80, 0xbb, 0xfb, 0xed, 0x9b, 0xc2, 0xd6, 0x05, 0x92}},
		{.data_size = 32, .data_offset = 0x160400, .ciphertext = {0x1e, 0x45, 0xab, 0xea, 0x70, 0x46, 0xb9, 0x08, 0x6d, 0x2f, 0xd1, 0xe4, 0x7f, 0xf3, 0x5d, 0xf9, 0x2e, 0xf9, 0x3d, 0x1f, 0x23, 0xe8, 0xa2, 0xd8, 0x5a, 0x53, 0xe7, 0xd7, 0xd7, 0x51, 0xe6, 0x92}},
	}
};

// For 64-byte k1 key
test_data_aes_mode_t test_data_xts_aes_256 = {
	.init_key = {0xee, 0x89, 0x95, 0xda, 0x3c, 0x8a, 0x43, 0x83, 0xa9, 0x4b, 0x25, 0x5b, 0x04, 0x7e, 0xf1, 0x57, 0xb8, 0xe8, 0x06, 0x45, 0x87, 0x76, 0xee, 0x1b, 0x4e, 0x2e, 0x55, 0xa7, 0x1f, 0x25, 0xe1, 0x94},
	.k2_info = {0x8f, 0x96, 0x33, 0x47, 0xe1, 0xa5, 0x57, 0xe9, 0x2a, 0x51, 0xa9, 0xbe, 0x48, 0x84, 0x25, 0x4e, 0x6f, 0x50, 0x1c, 0x45, 0xdb, 0xb6, 0xfa, 0xeb, 0x35, 0xd2, 0x27, 0x91, 0x3f, 0x67, 0x57, 0xd9, 0xcb, 0x55, 0xe4, 0x2b, 0x18, 0x16, 0xe7, 0xce, 0x6c, 0xf2, 0x58, 0x71, 0x17, 0x76, 0x2a, 0x86, 0x05, 0xe7, 0x37, 0x45, 0x71, 0x34, 0xca, 0xaf, 0x60, 0x07, 0xdf, 0xf4, 0xd2, 0xee, 0x3d, 0x4b},
	.k1_encrypted = {{0x37, 0xcf, 0x5b, 0x9e, 0x08, 0x26, 0x36, 0x31, 0xd7, 0x51, 0x3c, 0x33, 0x0d, 0x5d, 0x03, 0xad, 0x48, 0x6e, 0xbe, 0x82, 0xce, 0xa9, 0xc8, 0xd5, 0x98, 0x11, 0x24, 0xcc, 0x83, 0xf8, 0xf9, 0x53}, {0x84, 0xf7, 0x09, 0x06, 0xa3, 0xf2, 0xc7, 0x5f, 0x08, 0x43, 0xfd, 0xe9, 0x2e, 0xab, 0x32, 0xf3, 0x31, 0xd4, 0x4f, 0xf4, 0xf6, 0x1d, 0xa1, 0xc7, 0x1f, 0x2c, 0x11, 0xca, 0x9f, 0x21, 0x26, 0xaa}},
	.plaintext_data = {0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80},
    .xts_test_data = {
		{.data_size = 32, .data_offset = 0x160000, .ciphertext = {0x3d, 0xab, 0x5b, 0x96, 0x8f, 0xa5, 0x5b, 0x6f, 0xcb, 0x82, 0xcb, 0x3a, 0x94, 0xa0, 0xa6, 0x3e, 0xc6, 0x59, 0xde, 0x61, 0x0e, 0xb4, 0x8f, 0x99, 0x03, 0xff, 0xaf, 0x24, 0x11, 0x6e, 0x3b, 0x3b}},
		{.data_size = 64, .data_offset = 0x160100, .ciphertext = {0x75, 0xaa, 0x68, 0xcd, 0x98, 0x29, 0x70, 0xe2, 0xaf, 0xb2, 0x7c, 0x39, 0x0a, 0x50, 0xac, 0x08, 0x01, 0x71, 0x7a, 0xdb, 0x1f, 0x04, 0xb8, 0x5e, 0x4d, 0x8d, 0x4c, 0xc4, 0x0d, 0xde, 0xcf, 0x5e, 0x1f, 0xa0, 0xfb, 0x74, 0xf6, 0xb2, 0x51, 0xe0, 0x69, 0x84, 0x47, 0x61, 0x71, 0xc5, 0x92, 0x8c, 0x12, 0x69, 0x77, 0xfb, 0xdf, 0xe8, 0xd2, 0x86, 0xc3, 0xbc, 0xd2, 0x2d, 0x8c, 0xbc, 0x4b, 0x59}},
		{.data_size = 128, .data_offset = 0x160200, .ciphertext = {0xb9, 0x2b, 0xf5, 0x02, 0x84, 0xba, 0x1b, 0xda, 0xf8, 0x85, 0x2c, 0xba, 0x36, 0x42, 0x16, 0xa7, 0x83, 0x22, 0x37, 0xab, 0x41, 0x9d, 0x84, 0xde, 0x81, 0x4c, 0xa5, 0x2e, 0x40, 0x65, 0xd2, 0xc9, 0x30, 0x81, 0x5c, 0x61, 0x86, 0x22, 0xc3, 0xec, 0xf7, 0x7f, 0x66, 0x02, 0x73, 0x7b, 0xf2, 0x22, 0x17, 0x6d, 0x41, 0x90, 0xda, 0xb1, 0x4e, 0x2f, 0xbe, 0xe2, 0x0b, 0x22, 0x4e, 0xb4, 0x6a, 0x90, 0x13, 0x1f, 0x9b, 0xf2, 0x78, 0x03, 0x4f, 0xb5, 0x37, 0x62, 0x6d, 0x01, 0x56, 0xfc, 0xe3, 0xfa, 0xe7, 0x19, 0xbf, 0x81, 0x05, 0x9d, 0x7f, 0xef, 0xbe, 0xe4, 0x3a, 0xa8, 0xa0, 0x98, 0x74, 0x68, 0x10, 0xe4, 0x95, 0x7a, 0x93, 0xf0, 0x75, 0x68, 0x09, 0xf3, 0x63, 0xe9, 0x1a, 0xbb, 0x2a, 0xf4, 0x1e, 0xa0, 0x45, 0x61, 0x5b, 0x1f, 0x2d, 0x08, 0xfd, 0xed, 0x79, 0x61, 0x11, 0xad, 0x57, 0xb6}},
		{.data_size = 16, .data_offset = 0x160300, .ciphertext = {0x0f, 0x8f, 0xff, 0x33, 0x23, 0x1c, 0x63, 0x5f, 0xae, 0x91, 0x87, 0x13, 0x5d, 0x27, 0xd0, 0xef}},
		{.data_size = 32, .data_offset = 0x160400, .ciphertext = {0x29, 0x23, 0x17, 0x47, 0xb3, 0xd3, 0x19, 0xf4, 0x9d, 0x6b, 0x3c, 0x82, 0x49, 0x83, 0x25, 0x7a, 0x3d, 0xfc, 0x51, 0xca, 0x7c, 0x96, 0x5e, 0x1e, 0xcf, 0x7a, 0x85, 0x99, 0x0c, 0x33, 0x17, 0xe4}},
	}
};

test_data_aes_mode_t test_data_ecdsa = {
    .init_key = { 0xee, 0x89, 0x95, 0xda, 0x3c, 0x8a, 0x43, 0x83, 0xa9, 0x4b, 0x25, 0x5b, 0x04, 0x7e, 0xf1, 0x57, 0xb8, 0xe8, 0x06, 0x45, 0x87, 0x76, 0xee, 0x1b, 0x4e, 0x2e, 0x55, 0xa7, 0x1f, 0x25, 0xe1, 0x94 },
    .k2_info = { 0x8f, 0x96, 0x33, 0x47, 0xe1, 0xa5, 0x57, 0xe9, 0x2a, 0x51, 0xa9, 0xbe, 0x48, 0x84, 0x25, 0x4e, 0x6f, 0x50, 0x1c, 0x45, 0xdb, 0xb6, 0xfa, 0xeb, 0x35, 0xd2, 0x27, 0x91, 0x3f, 0x67, 0x57, 0xd9, 0xcb, 0x55, 0xe4, 0x2b, 0x18, 0x16, 0xe7, 0xce, 0x6c, 0xf2, 0x58, 0x71, 0x17, 0x76, 0x2a, 0x86, 0x05, 0xe7, 0x37, 0x45, 0x71, 0x34, 0xca, 0xaf, 0x60, 0x07, 0xdf, 0xf4, 0xd2, 0xee, 0x3d, 0x4b },
    .k1_encrypted = { { 0xe0, 0xe8, 0x41, 0xe3, 0xd0, 0x92, 0x71, 0x84, 0x4b, 0x02, 0x1e, 0xec, 0x14, 0xdd, 0xaf, 0xf8, 0x39, 0xf9, 0x6a, 0x8d, 0x1b, 0xd7, 0x64, 0x3b, 0x7b, 0xa6, 0x05, 0x42, 0x01, 0xfb, 0xab, 0xe1 }, {  } },
    .ecdsa_test_data = {
        .pubx = { 0x25, 0x8c, 0x48, 0x4d, 0x0b, 0x4d, 0x3f, 0xbf, 0xde, 0xcf, 0x00, 0xc9, 0x4b, 0x0b, 0xf1, 0x14, 0xb4, 0x31, 0x97, 0x79, 0x5a, 0xd3, 0x48, 0x72, 0x44, 0x2d, 0xab, 0x76, 0x29, 0xb9, 0x8b, 0x05 },
        .puby = { 0xf5, 0x6b, 0xfb, 0xb4, 0xe4, 0xde, 0x81, 0x83, 0xa7, 0x0a, 0x90, 0xe4, 0x33, 0x41, 0x92, 0xaa, 0xc5, 0xed, 0x93, 0xe0, 0x76, 0x2b, 0xe2, 0x4b, 0xdd, 0xa2, 0x8e, 0xe1, 0xc9, 0xe2, 0x94, 0x50 }
    }
};


test_data_ecdh0_mode_t test_data_ecdh0 = {
    .plaintext_data = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80 },
    .k1 = {
        { 0x2c, 0x92, 0xb7, 0xf5, 0x42, 0x10, 0xf1, 0x62, 0xf7, 0xad, 0x82, 0x3a, 0xe4, 0x03, 0x13, 0xe7, 0x17, 0x0f, 0x70, 0x73, 0xf1, 0x78, 0xd9, 0xcb, 0x63, 0xb9, 0x48, 0xcd, 0x4b, 0xb0, 0x7b, 0x06 },
        { 0x2c, 0x92, 0xb7, 0xf5, 0x42, 0x10, 0xf1, 0x62, 0xf7, 0xad, 0x82, 0x3a, 0xe4, 0x03, 0x13, 0xe7, 0x17, 0x0f, 0x70, 0x73, 0xf1, 0x78, 0xd9, 0xcb, 0x63, 0xb9, 0x48, 0xcd, 0x4b, 0xb0, 0x7b, 0x06 },
    },
    .k1_G = {
        { 0x25, 0x8c, 0x48, 0x4d, 0x0b, 0x4d, 0x3f, 0xbf, 0xde, 0xcf, 0x00, 0xc9, 0x4b, 0x0b, 0xf1, 0x14, 0xb4, 0x31, 0x97, 0x79, 0x5a, 0xd3, 0x48, 0x72, 0x44, 0x2d, 0xab, 0x76, 0x29, 0xb9, 0x8b, 0x05, 0xf5, 0x6b, 0xfb, 0xb4, 0xe4, 0xde, 0x81, 0x83, 0xa7, 0x0a, 0x90, 0xe4, 0x33, 0x41, 0x92, 0xaa, 0xc5, 0xed, 0x93, 0xe0, 0x76, 0x2b, 0xe2, 0x4b, 0xdd, 0xa2, 0x8e, 0xe1, 0xc9, 0xe2, 0x94, 0x50 },
        { 0x25, 0x8c, 0x48, 0x4d, 0x0b, 0x4d, 0x3f, 0xbf, 0xde, 0xcf, 0x00, 0xc9, 0x4b, 0x0b, 0xf1, 0x14, 0xb4, 0x31, 0x97, 0x79, 0x5a, 0xd3, 0x48, 0x72, 0x44, 0x2d, 0xab, 0x76, 0x29, 0xb9, 0x8b, 0x05, 0xf5, 0x6b, 0xfb, 0xb4, 0xe4, 0xde, 0x81, 0x83, 0xa7, 0x0a, 0x90, 0xe4, 0x33, 0x41, 0x92, 0xaa, 0xc5, 0xed, 0x93, 0xe0, 0x76, 0x2b, 0xe2, 0x4b, 0xdd, 0xa2, 0x8e, 0xe1, 0xc9, 0xe2, 0x94, 0x50 },
    }
};


test_data_aes_mode_t test_data_hmac = {
    .init_key = { 0xee, 0x89, 0x95, 0xda, 0x3c, 0x8a, 0x43, 0x83, 0xa9, 0x4b, 0x25, 0x5b, 0x04, 0x7e, 0xf1, 0x57, 0xb8, 0xe8, 0x06, 0x45, 0x87, 0x76, 0xee, 0x1b, 0x4e, 0x2e, 0x55, 0xa7, 0x1f, 0x25, 0xe1, 0x94 },
    .k2_info = { 0x8f, 0x96, 0x33, 0x47, 0xe1, 0xa5, 0x57, 0xe9, 0x2a, 0x51, 0xa9, 0xbe, 0x48, 0x84, 0x25, 0x4e, 0x6f, 0x50, 0x1c, 0x45, 0xdb, 0xb6, 0xfa, 0xeb, 0x35, 0xd2, 0x27, 0x91, 0x3f, 0x67, 0x57, 0xd9, 0xcb, 0x55, 0xe4, 0x2b, 0x18, 0x16, 0xe7, 0xce, 0x6c, 0xf2, 0x58, 0x71, 0x17, 0x76, 0x2a, 0x86, 0x05, 0xe7, 0x37, 0x45, 0x71, 0x34, 0xca, 0xaf, 0x60, 0x07, 0xdf, 0xf4, 0xd2, 0xee, 0x3d, 0x4b },
    .k1_encrypted = { { 0xd8, 0xf5, 0xe3, 0x3e, 0x9e, 0x79, 0xb7, 0x94, 0x3c, 0x84, 0xb0, 0xd4, 0x73, 0x21, 0x55, 0x39, 0x3f, 0xa4, 0x5f, 0x27, 0x5d, 0x4a, 0x2d, 0x2a, 0x30, 0xe5, 0xa2, 0xae, 0x78, 0xde, 0x34, 0x50 }, {  } },
    .hmac_test_data = {
        .message = { 0x44, 0x65, 0x6c, 0x65, 0x6e, 0x69, 0x74, 0x69, 0x20, 0x76, 0x6f, 0x6c, 0x75, 0x70, 0x74, 0x61, 0x73, 0x20, 0x65, 0x78, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x62, 0x6f, 0x20, 0x65, 0x74, 0x20, 0x61, 0x73, 0x73, 0x75, 0x6d, 0x65, 0x6e, 0x64, 0x61, 0x2e, 0x20, 0x53, 0x65, 0x64, 0x20, 0x65, 0x74, 0x20, 0x61, 0x6c, 0x69, 0x71, 0x75, 0x69, 0x64, 0x20, 0x6d, 0x69, 0x6e, 0x75, 0x73, 0x20, 0x71, 0x75, 0x69, 0x73, 0x2e, 0x20, 0x50, 0x72, 0x61, 0x65, 0x73, 0x65, 0x6e, 0x74, 0x69, 0x75, 0x6d, 0x20, 0x63, 0x75, 0x70, 0x69, 0x64, 0x69, 0x74, 0x61, 0x74, 0x65, 0x20, 0x71, 0x75, 0x69, 0x61, 0x20, 0x6e, 0x65, 0x6d, 0x6f, 0x20, 0x65, 0x73, 0x74, 0x2e, 0x20, 0x4c, 0x61, 0x62, 0x6f, 0x72, 0x69, 0x6f, 0x73, 0x61, 0x6d, 0x20, 0x70, 0x61, 0x72, 0x69, 0x61, 0x74, 0x75, 0x72, 0x20, 0x75, 0x74, 0x20, 0x64, 0x69, 0x73, 0x74, 0x69, 0x6e, 0x63, 0x74, 0x69, 0x6f, 0x20, 0x74, 0x65, 0x6e, 0x65, 0x74, 0x75, 0x72, 0x2e, 0x20, 0x53, 0x75, 0x6e, 0x74, 0x20, 0x61, 0x72, 0x63, 0x68, 0x69, 0x74, 0x65, 0x63, 0x74, 0x6f, 0x20, 0x69, 0x75, 0x72, 0x65, 0x20, 0x61, 0x73, 0x70, 0x65, 0x72, 0x6e, 0x61, 0x74, 0x75, 0x72, 0x20, 0x73, 0x6f, 0x6c, 0x75, 0x74, 0x61, 0x20, 0x75, 0x74, 0x20, 0x72, 0x65, 0x63, 0x75, 0x73, 0x61, 0x6e, 0x64, 0x61, 0x65, 0x2e, 0x20, 0x55, 0x74, 0x20, 0x71, 0x75, 0x69, 0x62, 0x75, 0x73, 0x64, 0x61, 0x6d, 0x20, 0x6f, 0x63, 0x63, 0x61, 0x65, 0x63, 0x61, 0x74, 0x69, 0x20, 0x75, 0x74, 0x20, 0x71, 0x75, 0x69, 0x20, 0x73, 0x69, 0x74, 0x20, 0x64, 0x69, 0x67, 0x6e, 0x69, 0x73, 0x73, 0x69, 0x6d, 0x6f, 0x73, 0x20, 0x65, 0x61, 0x71, 0x75, 0x65, 0x2e, 0x2e },
        .hmac_result = { 0xa8, 0xc0, 0x4e, 0x46, 0x70, 0x24, 0x52, 0x24, 0x47, 0x05, 0x8a, 0xa0, 0x99, 0x2b, 0xf8, 0x67, 0xf6, 0x72, 0x6f, 0x51, 0xe0, 0x94, 0x97, 0xe5, 0x88, 0x71, 0x2d, 0x42, 0x63, 0xa9, 0x2c, 0xb7 }
    }
};


test_data_aes_mode_t test_data_ds = {
    .init_key = { 0xee, 0x89, 0x95, 0xda, 0x3c, 0x8a, 0x43, 0x83, 0xa9, 0x4b, 0x25, 0x5b, 0x04, 0x7e, 0xf1, 0x57, 0xb8, 0xe8, 0x06, 0x45, 0x87, 0x76, 0xee, 0x1b, 0x4e, 0x2e, 0x55, 0xa7, 0x1f, 0x25, 0xe1, 0x94 },
    .k2_info = { 0x8f, 0x96, 0x33, 0x47, 0xe1, 0xa5, 0x57, 0xe9, 0x2a, 0x51, 0xa9, 0xbe, 0x48, 0x84, 0x25, 0x4e, 0x6f, 0x50, 0x1c, 0x45, 0xdb, 0xb6, 0xfa, 0xeb, 0x35, 0xd2, 0x27, 0x91, 0x3f, 0x67, 0x57, 0xd9, 0xcb, 0x55, 0xe4, 0x2b, 0x18, 0x16, 0xe7, 0xce, 0x6c, 0xf2, 0x58, 0x71, 0x17, 0x76, 0x2a, 0x86, 0x05, 0xe7, 0x37, 0x45, 0x71, 0x34, 0xca, 0xaf, 0x60, 0x07, 0xdf, 0xf4, 0xd2, 0xee, 0x3d, 0x4b },
    .k1_encrypted = { { 0xe0, 0xe8, 0x41, 0xe3, 0xd0, 0x92, 0x71, 0x84, 0x4b, 0x02, 0x1e, 0xec, 0x14, 0xdd, 0xaf, 0xf8, 0x39, 0xf9, 0x6a, 0x8d, 0x1b, 0xd7, 0x64, 0x3b, 0x7b, 0xa6, 0x05, 0x42, 0x01, 0xfb, 0xab, 0xe1 }, {  } },
    .ds_test_data = {
        .ds_message = { 0x61, 0xaf, 0x8a, 0x89, 0xd9, 0x0c, 0x47, 0x6f, 0x08, 0xf8, 0x39, 0x77, 0x6c, 0xf1, 0x24, 0x23, 0xaa, 0xe3, 0xa2, 0x68, 0x62, 0xa5, 0x45, 0x3b, 0x01, 0x1a, 0xc4, 0xf6, 0x0d, 0x20, 0x9d, 0x7e, 0xa9, 0xcd, 0x0a, 0x73, 0xc7, 0x27, 0xf6, 0xff, 0xf7, 0x97, 0x8d, 0xd3, 0x99, 0xbd, 0xcc, 0xdf, 0xa2, 0xc8, 0xf5, 0x84, 0x58, 0xce, 0xaf, 0xef, 0x62, 0x52, 0xfb, 0x0b, 0x3b, 0xd0, 0x55, 0xb4, 0x69, 0x2d, 0x33, 0xf4, 0xd9, 0x6f, 0x84, 0xba, 0x35, 0x98, 0x07, 0x13, 0x41, 0x2f, 0xab, 0xec, 0x9d, 0xf0, 0x69, 0xc5, 0x72, 0xde, 0x19, 0x0f, 0xac, 0xda, 0x9c, 0x89, 0x97, 0xfa, 0x6f, 0x88, 0xff, 0xc5, 0x14, 0x36, 0x7f, 0x53, 0x85, 0xcd, 0x07, 0x73, 0x52, 0xcc, 0xe2, 0x5a, 0xc3, 0x09, 0xc3, 0x4c, 0x3b, 0x9d, 0x9e, 0xaa, 0xe1, 0x69, 0x4a, 0xea, 0x38, 0xac, 0x38, 0x13, 0x50, 0x25, 0x4f, 0xe2, 0x01, 0x27, 0x82, 0xaa, 0x5e, 0x9c, 0xeb, 0x0b, 0xe1, 0x2b, 0x9a, 0x05, 0x71, 0xdd, 0xf5, 0x6b, 0xaa, 0x63, 0xdb, 0xa3, 0xbc, 0xff, 0x31, 0xf9, 0xdb, 0x1d, 0xe5, 0x5f, 0x82, 0xc3, 0x67, 0x9a, 0xee, 0x2b, 0xc6, 0xc6, 0xbb, 0x10, 0x2d, 0xa7, 0xa5, 0x16, 0x7b, 0x7a, 0x80, 0x81, 0xd2, 0xf3, 0xfd, 0x00, 0xd4, 0x8f, 0x7e, 0x5d, 0xba, 0xd5, 0xbd, 0x73, 0x51, 0x4a, 0x25, 0x13, 0x18, 0x3f, 0xbf, 0xd1, 0x4d, 0xc1, 0x23, 0x94, 0xc7, 0xb6, 0xd4, 0xa8, 0xd3, 0xa2, 0xcf, 0x7f, 0x0e, 0x1e, 0xa7, 0xd9, 0xce, 0x36, 0x3b, 0x04, 0x5f, 0xf8, 0x83, 0xc4, 0x21, 0xa0, 0x48, 0x3d, 0xe3, 0xc7, 0xf8, 0x41, 0x4e, 0x69, 0x18, 0x30, 0x4b, 0x36, 0x88, 0x9d, 0x33, 0x29, 0x7e, 0xe0, 0x6f, 0x93, 0x59, 0xa2, 0x0e, 0x50, 0xc9, 0x1d, 0x38, 0xe2, 0x1b, 0x68, 0xfa, 0xc1, 0x2f, 0x29, 0x35, 0x02, 0x65, 0x1a, 0x2b, 0x36, 0x98, 0x00, 0x3f, 0xf3, 0x6d, 0xa3, 0xbf, 0xe3, 0xd8, 0x39, 0x6e, 0x30, 0x1e, 0xd2, 0xe9, 0x58, 0xb0, 0xb0, 0x60, 0xc0, 0x85, 0x0b, 0x8d, 0x47, 0xa2, 0xcb, 0x67, 0x8a, 0xe3, 0x13, 0xec, 0x2d, 0x6d, 0x01, 0xf6, 0x6c, 0x07, 0xab, 0xfe, 0xe5, 0x29, 0xe9, 0x65, 0x3e, 0x55, 0x7a, 0x76, 0xcb, 0x45, 0xa1, 0x42, 0xa8, 0xa3, 0x34, 0x67, 0xb8, 0xec, 0x61, 0x00, 0xa7, 0xfc, 0xd0, 0xc3, 0xdb, 0xc9, 0xf9, 0xa1, 0x46, 0xf5, 0x8a, 0xc2, 0x96, 0xf4, 0xae, 0x34, 0xf9, 0xd8, 0x86, 0x9f, 0x67, 0x57, 0x58, 0xba, 0x46, 0x1b, 0x2f, 0xdf, 0x0f, 0x9c, 0x49, 0x21, 0x16, 0x2f, 0xb2, 0xc1, 0xa6, 0xe4, 0x8e, 0x4f, 0x87, 0x3d, 0x8c, 0x29, 0x03, 0x0a, 0x81, 0xbf, 0x3f, 0xa4, 0xee, 0x99, 0xd1, 0xbc, 0x01, 0x40, 0xfb, 0xdf, 0x1e, 0xe3, 0x71, 0x58, 0x5a },
        .ds_encrypted_input_params = { 0x1c, 0x8e, 0xa5, 0x03, 0xc3, 0x5e, 0xc3, 0xcc, 0xf6, 0x52, 0xda, 0x20, 0x95, 0xff, 0xf7, 0xe5, 0x36, 0x45, 0x4e, 0xae, 0xf6, 0x41, 0xcc, 0xda, 0x73, 0xa5, 0xfd, 0xa4, 0x56, 0x61, 0xaa, 0x5d, 0xf7, 0x55, 0xed, 0x01, 0x03, 0xd0, 0x15, 0x65, 0x32, 0x76, 0x38, 0xc4, 0x84, 0x25, 0x4f, 0x48, 0xe2, 0xe3, 0x1f, 0xf9, 0x16, 0xe2, 0xf2, 0x58, 0xde, 0x46, 0x2a, 0x92, 0x68, 0xb6, 0x1c, 0xe0, 0xde, 0x17, 0x77, 0x25, 0x06, 0xa2, 0x93, 0x02, 0xfe, 0x84, 0x72, 0xae, 0xba, 0x09, 0x45, 0xdc, 0x1e, 0x69, 0xa8, 0xbd, 0xfa, 0x34, 0xce, 0x58, 0x8c, 0x3a, 0x46, 0xab, 0xdc, 0x42, 0xa1, 0x54, 0x0d, 0x36, 0x91, 0x99, 0xe5, 0x27, 0x3c, 0x2b, 0x09, 0x7a, 0xc7, 0x79, 0xc0, 0x0f, 0x54, 0x02, 0xfa, 0x93, 0x95, 0xb0, 0xdd, 0xca, 0x26, 0xdc, 0x91, 0x65, 0x29, 0xf7, 0x19, 0xb0, 0xa8, 0xb3, 0xf8, 0xa4, 0xf8, 0xbc, 0x02, 0x84, 0x3d, 0xda, 0xef, 0x44, 0x48, 0xef, 0x06, 0xd8, 0x41, 0x44, 0x0d, 0x5c, 0xfa, 0x4c, 0xe2, 0x1a, 0x6e, 0x15, 0xf0, 0x74, 0xce, 0xfd, 0x61, 0xda, 0x83, 0xe4, 0x3a, 0xfc, 0xd3, 0xc4, 0xc2, 0x94, 0xb0, 0xf6, 0x80, 0x10, 0xf3, 0x6c, 0xa1, 0x3f, 0xe0, 0x55, 0x8e, 0x6d, 0x0c, 0x21, 0x3f, 0xfb, 0xe8, 0x61, 0x63, 0x01, 0x3f, 0x64, 0x93, 0xfc, 0x67, 0x0e, 0x0f, 0x85, 0x18, 0xe9, 0x34, 0x0b, 0x89, 0xa9, 0x87, 0x6f, 0xfe, 0x39, 0x91, 0x31, 0x42, 0xcd, 0x3c, 0xe0, 0x7b, 0x22, 0x46, 0xf2, 0xee, 0xb3, 0x4b, 0x5c, 0xd0, 0x6f, 0xb8, 0x0d, 0x24, 0x3d, 0x06, 0xf5, 0x13, 0x7c, 0x21, 0xb5, 0xc7, 0xd3, 0x0e, 0x9e, 0x53, 0x29, 0x19, 0x96, 0x76, 0xf3, 0x44, 0x69, 0x67, 0x3e, 0xb3, 0x3c, 0x89, 0x92, 0x6c, 0x86, 0xe3, 0x64, 0x5d, 0xc0, 0x8d, 0x7a, 0xfe, 0x01, 0x45, 0x98, 0x34, 0x3d, 0x26, 0x50, 0xd7, 0x33, 0xbb, 0xcf, 0xea, 0x53, 0x65, 0xb8, 0x4f, 0x51, 0x99, 0x91, 0x13, 0x32, 0xdd, 0x92, 0xd0, 0x22, 0xb2, 0xbd, 0x23, 0xed, 0xdf, 0x73, 0xea, 0x72, 0x02, 0x3d, 0x39, 0x38, 0x49, 0x18, 0x0a, 0xe7, 0x48, 0x96, 0xec, 0xce, 0x0c, 0x1a, 0x9f, 0xc5, 0x84, 0xfc, 0x95, 0x6c, 0x59, 0xb4, 0xda, 0x74, 0xb3, 0xd7, 0x94, 0xe4, 0x5c, 0xf3, 0x7f, 0xba, 0x17, 0x79, 0xf7, 0xdb, 0x1a, 0xd0, 0x51, 0xf5, 0xc6, 0xbd, 0x55, 0x1b, 0xcc, 0x67, 0x20, 0xc8, 0x70, 0x0d, 0xdc, 0x5d, 0x9d, 0xdd, 0x9e, 0x13, 0x6e, 0x4b, 0x07, 0x3b, 0x56, 0x34, 0x68, 0xe3, 0x9e, 0xa8, 0xa1, 0x46, 0xbb, 0x30, 0xb1, 0xc1, 0xef, 0xea, 0x86, 0xf7, 0x1f, 0x44, 0xee, 0xf5, 0x07, 0x73, 0x36, 0x78, 0xfd, 0x38, 0x43, 0xef, 0x7a, 0x93, 0x00, 0x4f, 0x6b, 0x69, 0xc5, 0x6a, 0x9d, 0xc7, 0x87, 0x99, 0xde, 0x19, 0xfa, 0x83, 0x65, 0x6e, 0x77, 0x5d, 0x09, 0x04, 0xb8, 0x7a, 0xfd, 0xc3, 0x6c, 0xb8, 0x48, 0xb4, 0x5a, 0x4b, 0x54, 0xb8, 0x52, 0xcc, 0x85, 0x9d, 0xd9, 0xa5, 0xa4, 0x49, 0x27, 0x56, 0x24, 0xfb, 0xc2, 0xf0, 0x81, 0x20, 0xa5, 0x00, 0xc7, 0xff, 0xd6, 0x9c, 0x08, 0x0f, 0xd4, 0xe4, 0x78, 0x96, 0xce, 0x53, 0xd5, 0x0b, 0xc3, 0x1a, 0xe5, 0xc9, 0xb2, 0x57, 0xea, 0x44, 0xd9, 0x74, 0x39, 0x6b, 0xd7, 0x73, 0x35, 0xd7, 0xb9, 0xa8, 0x25, 0x74, 0x90, 0xf2, 0x11, 0xd1, 0x65, 0xdc, 0xf7, 0x7d, 0x9a, 0x56, 0x01, 0x4b, 0x0d, 0x78, 0xe9, 0xd3, 0xe7, 0xae, 0x23, 0xe5, 0xa5, 0x1c, 0x8a, 0x0e, 0x95, 0xb6, 0xae, 0xdd, 0xdf, 0xcd, 0x89, 0xb7, 0x10, 0x54, 0x24, 0x69, 0x3e, 0xec, 0x66, 0x2b, 0x94, 0x3e, 0x6c, 0xcb, 0x70, 0x4c, 0xb9, 0xa8, 0x29, 0x97, 0xcd, 0xdb, 0x29, 0x3e, 0xd0, 0xbf, 0x5d, 0xa8, 0x55, 0x9d, 0x22, 0x3d, 0xb8, 0x84, 0xbc, 0xe2, 0xc4, 0x58, 0xe6, 0xbf, 0xd4, 0xd4, 0x99, 0xed, 0x85, 0x59, 0x5e, 0xb9, 0xc3, 0x8b, 0x78, 0x49, 0xd3, 0xea, 0xe3, 0x39, 0xb2, 0x5b, 0x37, 0x7a, 0xe9, 0x74, 0x14, 0x61, 0xb0, 0x05, 0x0f, 0x95, 0xca, 0xa6, 0xd7, 0x9e, 0xd6, 0x13, 0x7e, 0xda, 0xbc, 0x17, 0xee, 0x10, 0xcf, 0x10, 0xa4, 0x76, 0xd1, 0xff, 0xa7, 0xce, 0x5e, 0x66, 0xcd, 0x28, 0x12, 0x83, 0x73, 0x37, 0x59, 0x28, 0xac, 0x97, 0x3e, 0x82, 0xd6, 0x98, 0xd5, 0x34, 0xe2, 0xbf, 0xd8, 0xda, 0xad, 0x5e, 0x02, 0xdd, 0xe4, 0x50, 0xbb, 0x8f, 0x0d, 0x01, 0x36, 0x5c, 0xe0, 0x2d, 0xae, 0xa6, 0x09, 0xb9, 0xcf, 0x44, 0x36, 0x6b, 0x0c, 0xa9, 0x7f, 0x88, 0xd7, 0xd8, 0x6c, 0x98, 0xc9, 0x21, 0x70, 0xf7, 0x0d, 0x7e, 0x6d, 0x6b, 0x08, 0x09, 0xea, 0xbf, 0x95, 0xb1, 0xa3, 0x48, 0x73, 0xfc, 0xc0, 0xb8, 0x40, 0xee, 0xc6, 0xd0, 0xe1, 0xbb, 0x52, 0x95, 0x0e, 0xc7, 0xde, 0xc8, 0xe6, 0xcf, 0x51, 0x01, 0xfc, 0x69, 0xba, 0x75, 0x82, 0x32, 0x5f, 0x24, 0x5b, 0x60, 0xc0, 0x46, 0xc0, 0xc3, 0x07, 0x67, 0x25, 0x77, 0xaf, 0x2e, 0x4b, 0x88, 0xf5, 0xe6, 0xd7, 0x6b, 0xd3, 0x28, 0xa6, 0x92, 0x77, 0xc5, 0x02, 0x76, 0xbe, 0xe0, 0x70, 0x6e, 0xe8, 0xa0, 0x55, 0xb4, 0xfe, 0xfe, 0x8d, 0x57, 0x02, 0x80, 0xe3, 0x0e, 0xb4, 0x1b, 0xe0, 0xcb, 0xb2, 0xc8, 0x6c, 0xa9, 0x8a, 0xb8, 0x61, 0x9c, 0x87, 0x3e, 0x10, 0x25, 0x3e, 0x29, 0x68, 0x5b, 0x07, 0x7a, 0x29, 0xe6, 0x1b, 0x68, 0x24, 0x25, 0xbb, 0xfd, 0xec, 0xe0, 0xcf, 0x08, 0x20, 0x49, 0x36, 0x49, 0xdf, 0x6d, 0xb1, 0xf9, 0x17, 0xa2, 0xfb, 0xb3, 0xb0, 0x68, 0x1c, 0x2d, 0x4e, 0x8b, 0x95, 0xfa, 0x01, 0x1c, 0xea, 0x90, 0xd4, 0xf8, 0x3c, 0xcd, 0x17, 0xb0, 0x86, 0xd6, 0xbe, 0x41, 0xfa, 0x36, 0x7f, 0x0f, 0x12, 0xec, 0x3c, 0xb7, 0x38, 0x2e, 0xdb, 0x12, 0x0c, 0x6d, 0x88, 0x8e, 0x19, 0x97, 0x5c, 0xf4, 0xed, 0x11, 0x09, 0xfb, 0xda, 0x76, 0xe6, 0x5f, 0x16, 0x17, 0x43, 0xe6, 0x36, 0xab, 0x97, 0x36, 0x2d, 0x60, 0xf4, 0xf2, 0x94, 0xe3, 0x50, 0xe2, 0x42, 0x98, 0x32, 0x22, 0xde, 0x0f, 0x86, 0x83, 0x7d, 0xcf, 0x4a, 0x64, 0x91, 0xfc, 0x42, 0xea, 0xc2, 0xdf, 0x72, 0x33, 0xad, 0x8a, 0xc0, 0x20, 0x03, 0x32, 0xf8, 0x48, 0xdc, 0xe9, 0xed, 0x28, 0x34, 0x16, 0x84, 0xbc, 0xd2, 0xde, 0x47, 0x69, 0xf7, 0x65, 0x92, 0x28, 0x58, 0x30, 0x5e, 0xf9, 0xfe, 0x47, 0xff, 0x2d, 0xe3, 0xff, 0xf8, 0x37, 0xa5, 0xf2, 0x99, 0x7c, 0x86, 0xad, 0xfe, 0x92, 0x11, 0x19, 0x40, 0x45, 0x5b, 0x13, 0xaf, 0x87, 0x25, 0x74, 0x28, 0x8b, 0x34, 0x53, 0xcd, 0xc5, 0xb2, 0xde, 0x8c, 0x5a, 0xcc, 0x03, 0x2d, 0x88, 0x5e, 0xe3, 0x6a, 0x31, 0x14, 0x57, 0xdc, 0x9a, 0x05, 0x0a, 0xef, 0xa3, 0xb0, 0xcf, 0xda, 0xfe, 0xc1, 0x3b, 0x96, 0xb0, 0xa9, 0x0f, 0xdd, 0xbb, 0x7e, 0x71, 0xb8, 0x28, 0x0b, 0x79, 0x74, 0xf4, 0xcd, 0x70, 0x76, 0x1f, 0xcf, 0x60, 0x4d, 0x39, 0xb4, 0xac, 0x65, 0xc7, 0x23, 0x12, 0x04, 0xf1, 0xbf, 0x7f, 0xf1, 0xb5, 0x08, 0x87, 0x9b, 0x83, 0x15, 0x86, 0x3a, 0x19, 0x61, 0x87, 0x44, 0x19, 0xce, 0x8b, 0xac, 0xd3, 0xf2, 0x25, 0x75, 0x49, 0xc0, 0x0e, 0x97, 0x3b, 0x13, 0x12, 0x08, 0x83, 0xa0, 0x40, 0x48, 0x2d, 0x47, 0xb8, 0x7a, 0x68, 0xcd, 0x1d, 0x1d, 0xa8, 0x4f, 0xb4, 0x7a, 0x39, 0x0c, 0x13, 0xba, 0x38, 0x49, 0x33, 0x8e, 0x0e, 0x8c, 0x81, 0x16, 0x0c, 0x12, 0xb6, 0x1b, 0x07, 0x76, 0x67, 0xc1, 0x7a, 0xb7, 0x5a, 0x3b, 0x73, 0x5f, 0x6b, 0x96, 0x9c, 0xb4, 0x94, 0x25, 0x05, 0xaf, 0xb6, 0x8b, 0x2d, 0x4b, 0x82, 0x95, 0x88, 0xce, 0xc0, 0x31, 0x44, 0xa3, 0x69, 0x8d, 0x9d, 0xe4, 0x8b, 0xd8, 0x86, 0xd4, 0x63, 0x82, 0xe1, 0x98, 0x91, 0xb1, 0xd1, 0x6b, 0xf4, 0xb4, 0x79, 0xcf, 0xff, 0x2a, 0x5a, 0x6c, 0x50, 0x47, 0x83, 0x87, 0x90, 0xb8, 0x3b, 0xb1, 0xa6, 0x18, 0x33, 0x25, 0xc0, 0x8e, 0xbe, 0x27, 0x40, 0xea, 0x6f, 0x1e, 0x5c, 0x0b, 0x2c, 0x62, 0xf9, 0xf5, 0xf4, 0x4c, 0xf9, 0x46, 0x1a, 0x36, 0xe6, 0x74, 0x3f, 0x5e, 0x9c, 0x64, 0x33, 0xba, 0x4f, 0xc7, 0xee, 0x1e, 0x66, 0xf4, 0x29, 0xbe, 0xe0, 0x05, 0xbf, 0xad, 0x1b, 0x1b, 0xe4, 0x17, 0x2c, 0xe5, 0xf7, 0x44, 0xf1, 0xe6, 0x11, 0x05, 0x37, 0xa4, 0x5f, 0xd0, 0xaa, 0xb1, 0xc1, 0x49, 0x92, 0xf2, 0x8e, 0xc4, 0x37, 0xfe, 0x9f, 0xf4, 0x89, 0x22, 0x0c, 0x68, 0x06, 0xa9, 0x8e, 0x05, 0x04, 0x8a, 0xa9, 0x93, 0x18, 0xbe, 0x24, 0x0c, 0xc6, 0xbd, 0x6b, 0xa5, 0x1b, 0xbd, 0x6f, 0x78, 0x57, 0x13, 0xf0, 0xda, 0x23, 0x73, 0x1b, 0xf8, 0xcd, 0x08, 0x8c },
        .ds_encrypted_input_params_iv = { 0xbc, 0xea, 0xec, 0x15, 0x14, 0xa4, 0x09, 0x48, 0xc4, 0x3c, 0x64, 0xcb, 0xc9, 0x4b, 0x70, 0x32 },
        .ds_key_size = 3072,
        .ds_result = { 0xeb, 0x17, 0x4c, 0xa9, 0x8d, 0x88, 0x53, 0x06, 0x8b, 0x55, 0x84, 0xc7, 0xcf, 0x89, 0x26, 0x17, 0x02, 0x2a, 0xab, 0x5c, 0x1b, 0xa6, 0x22, 0x6b, 0xdb, 0x16, 0xb4, 0x17, 0x97, 0xef, 0x41, 0x85, 0x23, 0x30, 0x2c, 0xa5, 0x13, 0x45, 0x48, 0x8d, 0x53, 0x84, 0x5c, 0xff, 0x0c, 0x23, 0x40, 0x10, 0x56, 0x0f, 0xf5, 0xc1, 0x1e, 0xe0, 0x6e, 0x14, 0xf8, 0xc4, 0x25, 0xe5, 0x35, 0x7e, 0xd6, 0xcc, 0xff, 0x08, 0xe2, 0xc7, 0xb9, 0x03, 0x5f, 0x2c, 0xc9, 0xea, 0x24, 0xa1, 0x27, 0x24, 0x41, 0x49, 0x7c, 0x71, 0x27, 0x84, 0x34, 0x42, 0x74, 0xfc, 0x95, 0x34, 0x1a, 0xdb, 0x46, 0xab, 0x4e, 0x19, 0x8d, 0x77, 0x3a, 0xdb, 0x74, 0x4c, 0x48, 0x20, 0x97, 0xd7, 0x75, 0xcb, 0xa2, 0x28, 0xd2, 0xfd, 0x6c, 0x53, 0x61, 0x71, 0xe4, 0x96, 0x93, 0x9d, 0x98, 0x07, 0xd0, 0x1b, 0x9f, 0x17, 0x1c, 0xf1, 0x32, 0xc1, 0x13, 0x5a, 0x84, 0x49, 0x19, 0x19, 0xc5, 0x36, 0xb2, 0xdb, 0xf7, 0x8b, 0x8d, 0x39, 0xdd, 0xc6, 0xdb, 0x2d, 0xcf, 0xd2, 0xbe, 0x38, 0x8b, 0x90, 0x37, 0xeb, 0x1b, 0x70, 0xed, 0x94, 0x3b, 0xad, 0x5c, 0x19, 0xe2, 0xbe, 0x76, 0xa4, 0xb2, 0x60, 0xc9, 0xd2, 0x6f, 0xb4, 0x01, 0xb1, 0xd7, 0x37, 0xc7, 0xb2, 0xfd, 0xfa, 0xd0, 0xf7, 0xb8, 0x10, 0x2d, 0x59, 0xde, 0x4d, 0x08, 0x25, 0xd1, 0x79, 0x69, 0xfc, 0x87, 0x1a, 0xf1, 0x41, 0x8a, 0x13, 0x31, 0xc4, 0xd4, 0x36, 0x3c, 0x24, 0xc2, 0xe1, 0x27, 0x85, 0x72, 0xe4, 0x92, 0x7a, 0x89, 0x93, 0x47, 0xc6, 0x9b, 0xaa, 0xe0, 0x12, 0xc1, 0x4b, 0x7d, 0xd1, 0xc9, 0xc7, 0x9a, 0xab, 0xa8, 0x68, 0x4d, 0x9e, 0xf4, 0xc4, 0x2e, 0x4b, 0xe6, 0x6d, 0x2a, 0x07, 0xa1, 0xc1, 0x4b, 0x5f, 0x84, 0xf6, 0xd9, 0x5c, 0x15, 0xc8, 0xcc, 0xa5, 0x60, 0x1b, 0xc1, 0x83, 0x9d, 0x18, 0x82, 0xbd, 0x07, 0xb7, 0xa3, 0xb0, 0x93, 0x03, 0x8e, 0x9a, 0x95, 0x3f, 0xe1, 0x4c, 0x6c, 0x21, 0x7e, 0xb3, 0xd7, 0xae, 0xf9, 0x6f, 0x4e, 0x4c, 0xfa, 0xa0, 0x2a, 0x46, 0xae, 0x3a, 0xe6, 0x93, 0xae, 0x1a, 0xfb, 0xa5, 0x2f, 0xad, 0x92, 0xa5, 0x12, 0x76, 0xd6, 0xb9, 0x93, 0xbb, 0xa2, 0x2a, 0x01, 0x14, 0x3a, 0xd7, 0x17, 0xd6, 0xfd, 0xc5, 0x75, 0x9d, 0x8e, 0x4f, 0xee, 0x2f, 0x9f, 0xfb, 0x71, 0x9a, 0x7e, 0xd9, 0xff, 0x95, 0xa0, 0x9f, 0x3d, 0x83, 0x91, 0xcc, 0x86, 0xc2, 0x5c, 0x68, 0xd3, 0x9d, 0x7f, 0x71, 0x54, 0x7a, 0xec, 0xe6, 0x40, 0xa5, 0xe5, 0xb7, 0x84, 0xd7, 0x56, 0x6c, 0x40, 0xb3, 0xa1, 0x9a, 0x9e, 0x50, 0x2b, 0x93, 0xfe, 0xca, 0xa6, 0x3d, 0x02, 0x6b, 0x31, 0x51, 0x87, 0x9d, 0xa7, 0x30, 0x82, 0x49, 0xe5, 0x26, 0xc7, 0xaf }
    }
};
