/**
 * Check if commit messages are sufficiently descriptive (not too short).
 *
 * Search for commit messages that appear to be automatically generated by Gitlab or temporary messages and report them.
 *
 * @dangerjs WARN
 */

module.exports = function () {
    const mrCommits = danger.gitlab.commits;

    const detectRegexes = [
        /^Rebased.*onto.*/i, // Automatically generated message by Gitlab
        /^Fast-forwarded.*to.*/i, // Automatically generated message by Gitlab
        /^Applied suggestion to.*/i, // Automatically generated message by Gitlab
        /^Suggestion applied for line.*/i, // Automatically generated message by Gitlab
        /^Opened merge request/i, // Automatically generated message by Gitlab
        /^Merged.*/i, // Automatically generated message by Gitlab
        /^Opened issue #\d+:.*/i, // Automatically generated message by Gitlab
        /^Closed issue #\d+:.*/i, // Automatically generated message by Gitlab
        /^Tagged.*as.*/i, // Automatically generated message by Gitlab
        /^Deleted tag.*/i, // Automatically generated message by Gitlab
        /^WIP.*/i, // Message starts with prefix "WIP"
        /^Cleaned.*/i, // Message starts "Cleaned"
        /clean ?up/i, // Message contains "clean up"
        /^[^A-Za-z0-9\s].*/, // Message starts with special characters
    ];

    // Search for the messages in each commit
    let partMessages = [];
    for (const commit of mrCommits) {
        const commitMessage = commit.message;
        const commitMessageTitle = commit.title;

        // Check if the commit message contains any Jira ticket references
        const jiraTicketRegex = /[A-Z0-9]+-[0-9]+/g;
        const jiraTicketMatches = commitMessage.match(jiraTicketRegex);
        if (jiraTicketMatches) {
            const jiraTicketNames = jiraTicketMatches.join(", ");
            partMessages.push(
                `- the commit message \`${commitMessageTitle}\` probably contains Jira ticket reference (\`${jiraTicketNames}\`). Please remove Jira tickets from commit messages.`
            );
            continue;
        }

        // Check if the commit message matches any regex from "detectRegexes"
        if (detectRegexes.some((regex) => commitMessage.match(regex))) {
            partMessages.push(
                `- the commit message \`${commitMessageTitle}\` appears to be a temporary or automatically generated message`
            );
            continue;
        }

        // Check if the commit message is not too short
        const shortCommitMessageThreshold = 20; // commit message is considered too short below this number of characters
        if (commitMessage.length < shortCommitMessageThreshold) {
            partMessages.push(
                `- the commit message \`${commitMessageTitle}\` may not be sufficiently descriptive`
            );
        }
    }

	// Create report
	if (partMessages.length) {
		partMessages.sort();
		let dangerMessage = `\nSome issues found for the commit messages in this MR:\n${partMessages.join('\n')}
		\nPlease consider updating these commit messages. It is recommended to follow this [commit messages guide](https://gitlab.espressif.cn:6688/espressif/esp-idf/-/wikis/dev-proc/Commit-messages)`;
		warn(dangerMessage);
	}
};
